// VIVR Product Importer - Content Script
// Extracts product information from any e-commerce website

(function() {
  'use strict';

  // ============================================================
  // SCHEMA.ORG JSON-LD EXTRACTOR (universal, highest priority)
  // ============================================================
  function extractFromJsonLd() {
    const scripts = document.querySelectorAll('script[type="application/ld+json"]');
    for (const script of scripts) {
      try {
        let data = JSON.parse(script.textContent);
        // Handle arrays
        if (Array.isArray(data)) {
          data = data.find(d => d['@type'] === 'Product') || data[0];
        }
        // Handle @graph
        if (data['@graph']) {
          data = data['@graph'].find(d => d['@type'] === 'Product') || data['@graph'][0];
        }
        if (data && data['@type'] === 'Product') {
          const images = [];
          if (data.image) {
            const imgData = Array.isArray(data.image) ? data.image : [data.image];
            imgData.forEach(img => {
              if (typeof img === 'string') images.push(img);
              else if (img.contentUrl) images.push(img.contentUrl);
              else if (img.url) images.push(img.url);
            });
          }

          let price = null;
          if (data.offers) {
            const offer = Array.isArray(data.offers) ? data.offers[0] : data.offers;
            price = parseFloat(offer.price || offer.lowPrice) || null;
          }

          const title = data.name || '';
          const description = data.description || '';

          if (title) {
            return { title, description, price, images: images.slice(0, 10), source: 'json-ld' };
          }
        }
      } catch (e) { /* skip invalid JSON */ }
    }
    return null;
  }

  // ============================================================
  // SITE-SPECIFIC EXTRACTORS
  // ============================================================
  const extractors = {
    // Amazon
    amazon: {
      test: () => window.location.hostname.includes('amazon'),
      extract: () => ({
        title: document.querySelector('#productTitle')?.textContent?.trim() ||
               document.querySelector('.product-title-word-break')?.textContent?.trim(),
        description: document.querySelector('#productDescription')?.textContent?.trim() ||
                     document.querySelector('#feature-bullets')?.textContent?.trim() ||
                     document.querySelector('.a-expander-content')?.textContent?.trim(),
        price: extractAmazonPrice(),
        images: extractAmazonImages(),
        currency: 'EUR'
      })
    },

    // eBay
    ebay: {
      test: () => window.location.hostname.includes('ebay'),
      extract: () => ({
        title: document.querySelector('.x-item-title__mainTitle')?.textContent?.trim() ||
               document.querySelector('h1[itemprop="name"]')?.textContent?.trim(),
        description: document.querySelector('#desc_ifr')?.contentDocument?.body?.textContent?.trim() ||
                     document.querySelector('.x-item-description')?.textContent?.trim(),
        price: extractEbayPrice(),
        images: extractEbayImages(),
        currency: 'EUR'
      })
    },

    // AliExpress
    aliexpress: {
      test: () => window.location.hostname.includes('aliexpress'),
      extract: () => ({
        title: extractAliExpressTitle(),
        description: extractAliExpressDescription(),
        price: extractAliExpressPrice(),
        images: extractAliExpressImages(),
        currency: 'EUR'
      })
    },

    // IKEA
    ikea: {
      test: () => window.location.hostname.includes('ikea'),
      extract: () => {
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        return {
          title: document.querySelector('h1 .pip-header-section__title--big')?.textContent?.trim() ||
                 document.querySelector('.pip-header-section h1')?.textContent?.trim() ||
                 document.querySelector('h1')?.textContent?.trim(),
          description: document.querySelector('.pip-product-summary__description')?.textContent?.trim() ||
                       document.querySelector('[class*="product-details"] p')?.textContent?.trim(),
          price: parsePrice(document.querySelector('.pip-temp-price__integer')?.textContent ||
                            document.querySelector('.pip-price__integer')?.textContent),
          images: extractImagesFromSelectors([
            '.pip-media-grid__grid img',
            '.pip-image img',
            '[class*="product-image"] img'
          ], 'ikea.com'),
          currency: 'EUR'
        };
      }
    },

    // Maisons du Monde
    maisonsdumonde: {
      test: () => window.location.hostname.includes('maisonsdumonde'),
      extract: () => {
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        return {
          title: document.querySelector('h1[class*="product-title"]')?.textContent?.trim() ||
                 document.querySelector('.product-name h1')?.textContent?.trim() ||
                 document.querySelector('h1')?.textContent?.trim(),
          description: document.querySelector('[class*="product-description"]')?.textContent?.trim() ||
                       document.querySelector('.description-content')?.textContent?.trim() ||
                       document.querySelector('[itemprop="description"]')?.textContent?.trim(),
          price: parsePrice(document.querySelector('[class*="product-price"] [class*="current"]')?.textContent ||
                            document.querySelector('.price')?.textContent ||
                            document.querySelector('[itemprop="price"]')?.getAttribute('content')),
          images: extractImagesFromSelectors([
            '.product-gallery img',
            '.product-images img',
            '[class*="slider"] img',
            '[class*="carousel"] img',
            '[class*="gallery"] img'
          ], 'maisonsdumonde'),
          currency: 'EUR'
        };
      }
    },

    // La Redoute
    laredoute: {
      test: () => window.location.hostname.includes('laredoute'),
      extract: () => {
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        return {
          title: document.querySelector('h1[class*="productTitle"]')?.textContent?.trim() ||
                 document.querySelector('.product-title h1')?.textContent?.trim() ||
                 document.querySelector('h1')?.textContent?.trim(),
          description: document.querySelector('[class*="productDescription"]')?.textContent?.trim() ||
                       document.querySelector('.product-description')?.textContent?.trim() ||
                       document.querySelector('[itemprop="description"]')?.textContent?.trim(),
          price: parsePrice(document.querySelector('[class*="productPrice"]')?.textContent ||
                            document.querySelector('.price [class*="current"]')?.textContent ||
                            document.querySelector('[itemprop="price"]')?.getAttribute('content')),
          images: extractImagesFromSelectors([
            '[class*="productVisual"] img',
            '.product-image img',
            '[class*="gallery"] img',
            '[class*="slider"] img',
            '[class*="carousel"] img'
          ], 'laredoute'),
          currency: 'EUR'
        };
      }
    },

    // Leroy Merlin
    leroymerlin: {
      test: () => window.location.hostname.includes('leroymerlin'),
      extract: () => {
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        return {
          title: document.querySelector('h1[class*="product-header"]')?.textContent?.trim() ||
                 document.querySelector('.product-title')?.textContent?.trim() ||
                 document.querySelector('h1')?.textContent?.trim(),
          description: document.querySelector('[class*="product-description"]')?.textContent?.trim() ||
                       document.querySelector('.product-desc')?.textContent?.trim() ||
                       document.querySelector('[itemprop="description"]')?.textContent?.trim(),
          price: parsePrice(document.querySelector('[class*="product-price"]')?.textContent ||
                            document.querySelector('.price')?.textContent ||
                            document.querySelector('[itemprop="price"]')?.getAttribute('content')),
          images: extractImagesFromSelectors([
            '[class*="product-media"] img',
            '.product-gallery img',
            '[class*="slider"] img',
            '[class*="carousel"] img'
          ], 'leroymerlin'),
          currency: 'EUR'
        };
      }
    },

    // Conforama
    conforama: {
      test: () => window.location.hostname.includes('conforama'),
      extract: () => {
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        return {
          title: document.querySelector('.product-name h1')?.textContent?.trim() ||
                 document.querySelector('h1[class*="title"]')?.textContent?.trim() ||
                 document.querySelector('h1')?.textContent?.trim(),
          description: document.querySelector('.product-description')?.textContent?.trim() ||
                       document.querySelector('[class*="description"]')?.textContent?.trim(),
          price: parsePrice(document.querySelector('.product-price .price')?.textContent ||
                            document.querySelector('[class*="price"] [class*="current"]')?.textContent ||
                            document.querySelector('[itemprop="price"]')?.getAttribute('content')),
          images: extractImagesFromSelectors([
            '.product-gallery img',
            '.product-image img',
            '[class*="slider"] img',
            '[class*="gallery"] img'
          ], 'conforama'),
          currency: 'EUR'
        };
      }
    },

    // BUT
    but: {
      test: () => window.location.hostname.includes('but.fr'),
      extract: () => {
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        return {
          title: document.querySelector('h1[class*="product"]')?.textContent?.trim() ||
                 document.querySelector('.product-title')?.textContent?.trim() ||
                 document.querySelector('h1')?.textContent?.trim(),
          description: document.querySelector('[class*="product-description"]')?.textContent?.trim() ||
                       document.querySelector('[itemprop="description"]')?.textContent?.trim(),
          price: parsePrice(document.querySelector('[class*="product-price"]')?.textContent ||
                            document.querySelector('.price')?.textContent ||
                            document.querySelector('[itemprop="price"]')?.getAttribute('content')),
          images: extractImagesFromSelectors([
            '[class*="product-image"] img',
            '[class*="gallery"] img',
            '[class*="slider"] img',
            '[class*="carousel"] img'
          ], 'but.fr'),
          currency: 'EUR'
        };
      }
    },

    // Habitat
    habitat: {
      test: () => window.location.hostname.includes('habitat'),
      extract: () => {
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        return {
          title: document.querySelector('h1[class*="product"]')?.textContent?.trim() ||
                 document.querySelector('.product-name')?.textContent?.trim() ||
                 document.querySelector('h1')?.textContent?.trim(),
          description: document.querySelector('[class*="product-description"]')?.textContent?.trim() ||
                       document.querySelector('[itemprop="description"]')?.textContent?.trim(),
          price: parsePrice(document.querySelector('[class*="product-price"]')?.textContent ||
                            document.querySelector('.price')?.textContent ||
                            document.querySelector('[itemprop="price"]')?.getAttribute('content')),
          images: extractImagesFromSelectors([
            '.product-gallery img',
            '[class*="product-image"] img',
            '[class*="slider"] img'
          ], 'habitat'),
          currency: 'EUR'
        };
      }
    },

    // Westwing
    westwing: {
      test: () => window.location.hostname.includes('westwing'),
      extract: () => {
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        return {
          title: document.querySelector('h1[class*="product"]')?.textContent?.trim() ||
                 document.querySelector('[data-testid="product-name"]')?.textContent?.trim() ||
                 document.querySelector('h1')?.textContent?.trim(),
          description: document.querySelector('[class*="product-description"]')?.textContent?.trim() ||
                       document.querySelector('[data-testid="product-description"]')?.textContent?.trim(),
          price: parsePrice(document.querySelector('[data-testid="product-price"]')?.textContent ||
                            document.querySelector('[class*="price"]')?.textContent),
          images: extractImagesFromSelectors([
            '[class*="product-image"] img',
            '[class*="gallery"] img',
            '[data-testid*="image"] img',
            '[class*="slider"] img'
          ], 'westwing'),
          currency: 'EUR'
        };
      }
    },

    // Alinea
    alinea: {
      test: () => window.location.hostname.includes('alinea'),
      extract: () => {
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        return {
          title: document.querySelector('h1[class*="product"]')?.textContent?.trim() ||
                 document.querySelector('h1')?.textContent?.trim(),
          description: document.querySelector('[class*="product-description"]')?.textContent?.trim() ||
                       document.querySelector('[itemprop="description"]')?.textContent?.trim(),
          price: parsePrice(document.querySelector('[class*="price"] [class*="current"]')?.textContent ||
                            document.querySelector('.price')?.textContent ||
                            document.querySelector('[itemprop="price"]')?.getAttribute('content')),
          images: extractImagesFromSelectors([
            '.product-gallery img',
            '[class*="product-image"] img',
            '[class*="slider"] img'
          ], 'alinea'),
          currency: 'EUR'
        };
      }
    },

    // Kavehome
    kavehome: {
      test: () => window.location.hostname.includes('kavehome'),
      extract: () => {
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        return {
          title: document.querySelector('h1[class*="product"]')?.textContent?.trim() ||
                 document.querySelector('.product-name')?.textContent?.trim() ||
                 document.querySelector('h1')?.textContent?.trim(),
          description: document.querySelector('[class*="product-description"]')?.textContent?.trim() ||
                       document.querySelector('[class*="description"]')?.textContent?.trim(),
          price: parsePrice(document.querySelector('[class*="product-price"]')?.textContent ||
                            document.querySelector('.price')?.textContent),
          images: extractImagesFromSelectors([
            '.product-gallery img',
            '[class*="product-media"] img',
            '[class*="gallery"] img',
            '[class*="slider"] img'
          ], 'kavehome'),
          currency: 'EUR'
        };
      }
    },

    // Tikamoon
    tikamoon: {
      test: () => window.location.hostname.includes('tikamoon'),
      extract: () => {
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        return {
          title: document.querySelector('h1[class*="product"]')?.textContent?.trim() ||
                 document.querySelector('h1')?.textContent?.trim(),
          description: document.querySelector('[class*="product-description"]')?.textContent?.trim() ||
                       document.querySelector('[itemprop="description"]')?.textContent?.trim(),
          price: parsePrice(document.querySelector('[class*="product-price"]')?.textContent ||
                            document.querySelector('.price')?.textContent ||
                            document.querySelector('[itemprop="price"]')?.getAttribute('content')),
          images: extractImagesFromSelectors([
            '.product-gallery img',
            '[class*="product-image"] img',
            '[class*="slider"] img'
          ], 'tikamoon'),
          currency: 'EUR'
        };
      }
    },

    // Cdiscount
    cdiscount: {
      test: () => window.location.hostname.includes('cdiscount'),
      extract: () => ({
        title: document.querySelector('.fpDesCol h1')?.textContent?.trim() ||
               document.querySelector('[itemprop="name"]')?.textContent?.trim(),
        description: document.querySelector('.fpDescTxt')?.textContent?.trim(),
        price: extractCdiscountPrice(),
        images: extractCdiscountImages(),
        currency: 'EUR'
      })
    },

    // Fnac
    fnac: {
      test: () => window.location.hostname.includes('fnac'),
      extract: () => ({
        title: document.querySelector('.f-productHeader-Title')?.textContent?.trim(),
        description: document.querySelector('.f-productDetails-descriptionContent')?.textContent?.trim(),
        price: extractFnacPrice(),
        images: extractFnacImages(),
        currency: 'EUR'
      })
    },

    // Zara Home
    zarahome: {
      test: () => window.location.hostname.includes('zarahome'),
      extract: () => {
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        return {
          title: document.querySelector('h1[class*="product-detail"]')?.textContent?.trim() ||
                 document.querySelector('h1')?.textContent?.trim(),
          description: document.querySelector('[class*="product-detail-description"]')?.textContent?.trim() ||
                       document.querySelector('[class*="description"]')?.textContent?.trim(),
          price: parsePrice(document.querySelector('[class*="price"] [class*="current"]')?.textContent ||
                            document.querySelector('[data-qa="product-price"]')?.textContent),
          images: extractImagesFromSelectors([
            '[class*="media-image"] img',
            '[class*="product-detail"] img',
            'picture img'
          ], 'zarahome'),
          currency: 'EUR'
        };
      }
    },

    // H&M Home
    hmhome: {
      test: () => window.location.hostname.includes('hm.com') || window.location.hostname.includes('www2.hm.com'),
      extract: () => {
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        return {
          title: document.querySelector('h1[class*="product"]')?.textContent?.trim() ||
                 document.querySelector('h1')?.textContent?.trim(),
          description: document.querySelector('[class*="product-description"]')?.textContent?.trim(),
          price: parsePrice(document.querySelector('[class*="price"] [class*="value"]')?.textContent ||
                            document.querySelector('[id*="product-price"]')?.textContent),
          images: extractImagesFromSelectors([
            '[class*="product-detail"] img',
            '[class*="product-image"] img',
            'picture source'
          ], 'hm.com'),
          currency: 'EUR'
        };
      }
    },

    // Manomano
    manomano: {
      test: () => window.location.hostname.includes('manomano'),
      extract: () => {
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        return {
          title: document.querySelector('h1[class*="product"]')?.textContent?.trim() ||
                 document.querySelector('[data-testid="product-title"]')?.textContent?.trim() ||
                 document.querySelector('h1')?.textContent?.trim(),
          description: document.querySelector('[class*="product-description"]')?.textContent?.trim() ||
                       document.querySelector('[itemprop="description"]')?.textContent?.trim(),
          price: parsePrice(document.querySelector('[data-testid="product-price"]')?.textContent ||
                            document.querySelector('[class*="price"]')?.textContent),
          images: extractImagesFromSelectors([
            '[class*="product-image"] img',
            '[class*="gallery"] img',
            '[class*="slider"] img'
          ], 'manomano'),
          currency: 'EUR'
        };
      }
    },

    // Delamaison / Camif
    delamaison: {
      test: () => window.location.hostname.includes('delamaison') || window.location.hostname.includes('camif'),
      extract: () => {
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        return {
          title: document.querySelector('h1[class*="product"]')?.textContent?.trim() ||
                 document.querySelector('h1')?.textContent?.trim(),
          description: document.querySelector('[class*="product-description"]')?.textContent?.trim() ||
                       document.querySelector('[itemprop="description"]')?.textContent?.trim(),
          price: parsePrice(document.querySelector('[class*="price"] [class*="current"]')?.textContent ||
                            document.querySelector('[itemprop="price"]')?.getAttribute('content')),
          images: extractImagesFromSelectors([
            '.product-gallery img',
            '[class*="product-image"] img',
            '[class*="slider"] img'
          ], ''),
          currency: 'EUR'
        };
      }
    },

    // Maison HAND / Smallable / AM.PM / Drawer / Decoclico / Selency
    shopify: {
      test: () => {
        // Detect Shopify sites
        return !!document.querySelector('meta[name="shopify-checkout-api-token"]') ||
               !!document.querySelector('link[href*="cdn.shopify"]') ||
               !!document.querySelector('script[src*="cdn.shopify"]');
      },
      extract: () => {
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        // Shopify exposes product data in a global variable
        try {
          const shopifyProduct = window.ShopifyAnalytics?.meta?.product ||
                                 window.meta?.product;
          if (shopifyProduct) {
            return {
              title: shopifyProduct.name || document.querySelector('h1')?.textContent?.trim(),
              description: document.querySelector('[class*="product-description"]')?.textContent?.trim() ||
                           document.querySelector('.product__description')?.textContent?.trim(),
              price: shopifyProduct.variants?.[0]?.price ? parseFloat(shopifyProduct.variants[0].price) / 100 : null,
              images: (shopifyProduct.images || []).slice(0, 10),
              currency: 'EUR'
            };
          }
        } catch (e) {}
        return {
          title: document.querySelector('h1.product__title, h1[class*="product"]')?.textContent?.trim() ||
                 document.querySelector('h1')?.textContent?.trim(),
          description: document.querySelector('.product__description, [class*="product-description"]')?.textContent?.trim(),
          price: parsePrice(document.querySelector('.product__price, [class*="product-price"], .price')?.textContent),
          images: extractImagesFromSelectors([
            '.product__media img',
            '.product-gallery img',
            '.product-images img',
            '[class*="product-image"] img'
          ], ''),
          currency: 'EUR'
        };
      }
    },

    // WooCommerce (WordPress)
    woocommerce: {
      test: () => {
        return !!document.querySelector('body.woocommerce') ||
               !!document.querySelector('.woocommerce-product-gallery') ||
               !!document.querySelector('link[href*="woocommerce"]');
      },
      extract: () => {
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        return {
          title: document.querySelector('.product_title')?.textContent?.trim() ||
                 document.querySelector('h1')?.textContent?.trim(),
          description: document.querySelector('.woocommerce-product-details__short-description')?.textContent?.trim() ||
                       document.querySelector('[itemprop="description"]')?.textContent?.trim(),
          price: parsePrice(document.querySelector('.woocommerce-Price-amount')?.textContent ||
                            document.querySelector('.price ins .amount')?.textContent ||
                            document.querySelector('.price .amount')?.textContent),
          images: extractImagesFromSelectors([
            '.woocommerce-product-gallery img',
            '.wp-post-image',
            '.product-images img'
          ], ''),
          currency: 'EUR'
        };
      }
    },

    // PrestaShop (widely used in France)
    prestashop: {
      test: () => {
        return !!document.querySelector('#product-page, [id*="prestashop"]') ||
               !!document.querySelector('meta[name="generator"][content*="PrestaShop"]') ||
               !!document.querySelector('script[src*="prestashop"]') ||
               !!document.querySelector('.product-cover');
      },
      extract: () => {
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        return {
          title: document.querySelector('h1[itemprop="name"]')?.textContent?.trim() ||
                 document.querySelector('.product-title')?.textContent?.trim() ||
                 document.querySelector('h1')?.textContent?.trim(),
          description: document.querySelector('[itemprop="description"]')?.textContent?.trim() ||
                       document.querySelector('.product-description')?.textContent?.trim(),
          price: parsePrice(document.querySelector('[itemprop="price"]')?.getAttribute('content') ||
                            document.querySelector('.current-price [itemprop="price"]')?.textContent ||
                            document.querySelector('.product-price')?.textContent),
          images: extractImagesFromSelectors([
            '.product-cover img',
            '.product-images img',
            '.images-container img',
            '#product-images-large img'
          ], ''),
          currency: 'EUR'
        };
      }
    },

    // Generic extractor (fallback)
    generic: {
      test: () => true,
      extract: () => {
        // Try JSON-LD first for any site
        const jsonLd = extractFromJsonLd();
        if (jsonLd && jsonLd.title) return jsonLd;
        // Then try Open Graph meta tags
        const ogData = extractFromOpenGraph();
        if (ogData && ogData.title) return ogData;
        // Finally CSS selectors
        return {
          title: extractGenericTitle(),
          description: extractGenericDescription(),
          price: extractGenericPrice(),
          images: extractGenericImages(),
          currency: 'EUR'
        };
      }
    }
  };

  // ============================================================
  // OPEN GRAPH META TAGS EXTRACTOR
  // ============================================================
  function extractFromOpenGraph() {
    const title = document.querySelector('meta[property="og:title"]')?.getAttribute('content');
    const description = document.querySelector('meta[property="og:description"]')?.getAttribute('content');
    const image = document.querySelector('meta[property="og:image"]')?.getAttribute('content');
    const priceAmount = document.querySelector('meta[property="product:price:amount"]')?.getAttribute('content') ||
                        document.querySelector('meta[property="og:price:amount"]')?.getAttribute('content');

    if (title) {
      return {
        title: title.trim(),
        description: description?.trim() || '',
        price: priceAmount ? parseFloat(priceAmount) : null,
        images: image ? [image] : [],
        source: 'opengraph'
      };
    }
    return null;
  }

  // ============================================================
  // SHARED IMAGE EXTRACTION HELPER
  // ============================================================
  function extractImagesFromSelectors(selectors, domainHint) {
    const images = [];
    for (const selector of selectors) {
      document.querySelectorAll(selector).forEach(el => {
        let src = el.getAttribute('data-src') ||
                  el.getAttribute('data-lazy-src') ||
                  el.getAttribute('data-zoom-image') ||
                  el.getAttribute('data-full-image') ||
                  el.getAttribute('srcset')?.split(',')[0]?.trim()?.split(' ')[0] ||
                  el.getAttribute('src');
        if (src && isValidProductImage(src)) {
          images.push(cleanImageUrl(src));
        }
      });
    }

    // If no images found, try large images on the page
    if (images.length === 0) {
      document.querySelectorAll('img').forEach(img => {
        if ((img.naturalWidth >= 200 || img.width >= 200) && img.naturalHeight >= 200) {
          const src = img.getAttribute('src');
          if (src && isValidProductImage(src)) {
            images.push(cleanImageUrl(src));
          }
        }
      });
    }

    return [...new Set(images)].slice(0, 10);
  }

  // ============================================================
  // AMAZON HELPERS
  // ============================================================
  function extractAmazonPrice() {
    const priceWhole = document.querySelector('.a-price-whole')?.textContent?.trim();
    const priceFraction = document.querySelector('.a-price-fraction')?.textContent?.trim();
    if (priceWhole) {
      const price = parseFloat(`${priceWhole.replace(/[^0-9]/g, '')}.${priceFraction || '00'}`);
      return isNaN(price) ? null : price;
    }
    const priceText = document.querySelector('#priceblock_ourprice, #priceblock_dealprice, .a-price .a-offscreen')?.textContent;
    return parsePrice(priceText);
  }

  function extractAmazonImages() {
    const images = [];
    const mainImg = document.querySelector('#landingImage, #imgBlkFront');
    if (mainImg) {
      const src = mainImg.getAttribute('data-old-hires') || mainImg.getAttribute('src');
      if (src) images.push(cleanImageUrl(src));
    }
    document.querySelectorAll('#altImages img, .imageThumbnail img').forEach(img => {
      let src = img.getAttribute('src');
      if (src && !src.includes('icon') && !src.includes('sprite')) {
        src = src.replace(/\._[^.]+_\./, '.');
        if (!images.includes(src)) images.push(cleanImageUrl(src));
      }
    });
    return images.slice(0, 10);
  }

  // ============================================================
  // EBAY HELPERS
  // ============================================================
  function extractEbayPrice() {
    const priceEl = document.querySelector('.x-price-primary span, [itemprop="price"]');
    return parsePrice(priceEl?.textContent || priceEl?.getAttribute('content'));
  }

  function extractEbayImages() {
    const images = [];
    document.querySelectorAll('.ux-image-carousel-item img, .pic img').forEach(img => {
      const src = img.getAttribute('data-src') || img.getAttribute('src');
      if (src && src.includes('ebayimg')) {
        images.push(cleanImageUrl(src.replace(/s-l[0-9]+/, 's-l1600')));
      }
    });
    return [...new Set(images)].slice(0, 10);
  }

  // ============================================================
  // ALIEXPRESS HELPERS
  // ============================================================
  function extractAliExpressTitle() {
    const selectors = [
      'h1[data-pl="product-title"]', '.product-title-text',
      '[class*="ProductTitle"]', '[class*="product-title"]',
      '[class*="Title__text"]', '[class*="title--wrap"] h1',
      '.pdp-info h1', 'h1.pdp-mod-product-badge-title',
      '[data-spm="title"] h1', 'h1'
    ];
    for (const selector of selectors) {
      const el = document.querySelector(selector);
      const text = el?.textContent?.trim();
      if (text && text.length > 5 && text.length < 500) return text;
    }
    return document.title.split('|')[0].split('-')[0].trim();
  }

  function extractAliExpressDescription() {
    const selectors = [
      '.product-description', '.detail-desc-decorate-richtext',
      '[class*="ProductDescription"]', '[class*="description"]',
      '.pdp-product-desc', '[data-pl="product-description"]',
      '.product-overview', '[class*="Specification"]'
    ];
    for (const selector of selectors) {
      const el = document.querySelector(selector);
      const text = el?.textContent?.trim();
      if (text && text.length > 10) return text.substring(0, 2000);
    }
    return document.querySelector('meta[name="description"]')?.getAttribute('content') || '';
  }

  function extractAliExpressPrice() {
    const selectors = [
      '[data-pl="product-price"]', '.product-price-value',
      '[class*="Price__current"]', '[class*="price--current"]',
      '[class*="product-price"]', '.pdp-price',
      '[class*="uniformBannerBoxPrice"]',
      '.snow-price_SnowPrice__mainS__1bjar',
      '[class*="es--wrap"]', '.es--wrap--erdmPRe span'
    ];
    for (const selector of selectors) {
      const el = document.querySelector(selector);
      if (el) {
        const price = parsePrice(el.textContent);
        if (price && price > 0) return price;
      }
    }
    return null;
  }

  function extractAliExpressImages() {
    const images = [];
    const selectors = [
      '.images-view-item img', '.slider--img--D7MJNPZ img',
      '[class*="slider"] img', '[class*="gallery"] img',
      '[class*="magnifier"] img', '[class*="image-view"] img',
      '.product-image img', '[data-pl="product-image"] img',
      '.main-image img', '[class*="ImageView"] img'
    ];
    for (const selector of selectors) {
      document.querySelectorAll(selector).forEach(img => {
        let src = img.getAttribute('data-src') || img.getAttribute('src') || img.getAttribute('data-lazy-src');
        if (src && src.includes('alicdn')) {
          src = src.replace(/_[0-9]+x[0-9]+[^.]*/, '').replace(/_.webp/, '.jpg');
          if (!images.includes(src)) images.push(cleanImageUrl(src));
        }
      });
    }
    if (images.length === 0) {
      document.querySelectorAll('img').forEach(img => {
        const src = img.getAttribute('src') || img.getAttribute('data-src');
        if (src && src.includes('alicdn') && !src.includes('icon') && !src.includes('logo')) {
          if (img.naturalWidth >= 100 || img.width >= 100) images.push(cleanImageUrl(src));
        }
      });
    }
    return [...new Set(images)].slice(0, 10);
  }

  // ============================================================
  // CDISCOUNT / FNAC HELPERS
  // ============================================================
  function extractCdiscountPrice() {
    const priceEl = document.querySelector('.fpPrice, [itemprop="price"]');
    return parsePrice(priceEl?.textContent || priceEl?.getAttribute('content'));
  }

  function extractCdiscountImages() {
    const images = [];
    document.querySelectorAll('.fpImgMain img, .fpGalImg img').forEach(img => {
      const src = img.getAttribute('data-src') || img.getAttribute('src');
      if (src) images.push(cleanImageUrl(src));
    });
    return [...new Set(images)].slice(0, 10);
  }

  function extractFnacPrice() {
    const priceEl = document.querySelector('.f-priceBox-price, [itemprop="price"]');
    return parsePrice(priceEl?.textContent || priceEl?.getAttribute('content'));
  }

  function extractFnacImages() {
    const images = [];
    document.querySelectorAll('.f-productVisuals-mainMedia img').forEach(img => {
      const src = img.getAttribute('data-src') || img.getAttribute('src');
      if (src) images.push(cleanImageUrl(src));
    });
    return [...new Set(images)].slice(0, 10);
  }

  // ============================================================
  // GENERIC EXTRACTORS
  // ============================================================
  function extractGenericTitle() {
    const selectors = [
      'h1[itemprop="name"]', '[itemprop="name"]',
      '.product-title', '.product-name', '.product_title',
      '#product-title', '#product_title', 'h1.title',
      '.pdp-title', 'h1'
    ];
    for (const selector of selectors) {
      const el = document.querySelector(selector);
      if (el?.textContent?.trim()) return el.textContent.trim();
    }
    return document.title.split('|')[0].split('-')[0].trim();
  }

  function extractGenericDescription() {
    const selectors = [
      '[itemprop="description"]', '.product-description',
      '.product-desc', '#product-description', '.description',
      '.product-details', '.product-info', 'meta[name="description"]'
    ];
    for (const selector of selectors) {
      const el = document.querySelector(selector);
      if (el) {
        const content = el.tagName === 'META' ? el.getAttribute('content') : el.textContent;
        if (content?.trim()) return content.trim();
      }
    }
    return '';
  }

  function extractGenericPrice() {
    const selectors = [
      '[itemprop="price"]', '.price', '.product-price',
      '.current-price', '.sale-price', '.regular-price',
      '#product-price', '.offer-price', '[data-price]', '.price-current'
    ];
    for (const selector of selectors) {
      const el = document.querySelector(selector);
      if (el) {
        const price = parsePrice(el.textContent || el.getAttribute('content') || el.getAttribute('data-price'));
        if (price) return price;
      }
    }
    return null;
  }

  function extractGenericImages() {
    const images = [];
    const productSelectors = [
      '[itemprop="image"]', '.product-image img', '.product-gallery img',
      '.product-photo img', '#product-image img', '.main-image img',
      '.gallery img', '.swiper-slide img', '.carousel-item img'
    ];
    for (const selector of productSelectors) {
      document.querySelectorAll(selector).forEach(img => {
        const src = img.getAttribute('data-src') || img.getAttribute('data-lazy-src') ||
                    img.getAttribute('src') || img.getAttribute('content');
        if (src && isValidProductImage(src)) images.push(cleanImageUrl(src));
      });
    }
    if (images.length === 0) {
      document.querySelectorAll('img').forEach(img => {
        if (img.naturalWidth >= 200 && img.naturalHeight >= 200) {
          const src = img.getAttribute('src');
          if (src && isValidProductImage(src)) images.push(cleanImageUrl(src));
        }
      });
    }
    return [...new Set(images)].slice(0, 10);
  }

  // ============================================================
  // UTILITY FUNCTIONS
  // ============================================================
  function parsePrice(text) {
    if (!text) return null;
    const cleaned = text.replace(/[^\d.,]/g, '').trim();
    let price;
    if (cleaned.includes(',') && cleaned.includes('.')) {
      const commaPos = cleaned.lastIndexOf(',');
      const dotPos = cleaned.lastIndexOf('.');
      if (commaPos > dotPos) {
        price = parseFloat(cleaned.replace(/\./g, '').replace(',', '.'));
      } else {
        price = parseFloat(cleaned.replace(/,/g, ''));
      }
    } else if (cleaned.includes(',')) {
      price = parseFloat(cleaned.replace(',', '.'));
    } else {
      price = parseFloat(cleaned);
    }
    return isNaN(price) ? null : price;
  }

  function cleanImageUrl(url) {
    if (!url) return '';
    if (url.startsWith('//')) url = 'https:' + url;
    else if (url.startsWith('/')) url = window.location.origin + url;
    try {
      return new URL(url).href;
    } catch {
      return url;
    }
  }

  function isValidProductImage(src) {
    if (!src) return false;
    const invalidPatterns = [
      'logo', 'icon', 'sprite', 'banner', 'placeholder',
      'loading', 'pixel', 'tracking', 'ad-', 'advertisement',
      'avatar', 'profile', 'social', 'facebook', 'twitter',
      'instagram', 'pinterest', 'youtube', 'payment', 'card',
      'visa', 'mastercard', 'paypal', 'footer', 'header',
      'favicon', 'badge', 'flag', 'rating', 'star', 'arrow',
      'close', 'search', 'menu', 'cart', 'user', 'account'
    ];
    const srcLower = src.toLowerCase();
    return !invalidPatterns.some(pattern => srcLower.includes(pattern));
  }

  // ============================================================
  // MAIN EXTRACTION
  // ============================================================
  function extractProductData() {
    for (const [name, extractor] of Object.entries(extractors)) {
      if (name !== 'generic' && extractor.test()) {
        console.log(`[VIVR] Using ${name} extractor`);
        const data = extractor.extract();
        if (data && data.title) {
          return {
            ...data,
            sourceUrl: window.location.href,
            sourceDomain: window.location.hostname,
            extractedAt: new Date().toISOString()
          };
        }
      }
    }
    // Fallback to generic
    console.log('[VIVR] Using generic extractor');
    const data = extractors.generic.extract();
    return {
      ...data,
      sourceUrl: window.location.href,
      sourceDomain: window.location.hostname,
      extractedAt: new Date().toISOString()
    };
  }

  // Listen for messages from popup
  chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'extractProduct') {
      try {
        const productData = extractProductData();
        sendResponse({ success: true, data: productData });
      } catch (error) {
        console.error('[VIVR] Extraction error:', error);
        sendResponse({ success: false, error: error.message });
      }
    }
    return true;
  });

  console.log('[VIVR] Content script loaded');
})();
